/*
 * arch/mips/au1000/adm5120/ds1305.c -- Maxim/Dallas DS1305 RTC
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2004 MegaSolution Inc.
 * Author: m_sato@megasolution.jp
 */
#include <common.h>
#include <command.h>
#include <adm51xx_switch.h>
#include <asm/addrspace.h>
#include <asm/io.h>
#include <rtc.h>
#include <i2c.h>
#include <asm/adm5120.h>

#define ADDR_SEC	0x00
#define ADDR_MIN	0x01
#define ADDR_HOUR	0x02
#define ADDR_WDAY	0x03
#define ADDR_DAY	0x04
#define ADDR_MONTH	0x05
#define ADDR_YEAR	0x06
#define ADDR_ALARM0_SEC	0x07
#define ADDR_ALARM0_MIN	0x08
#define ADDR_ALARM0_HOUR	0x09
#define ADDR_ALARM0_WDAY	0x0a
#define ADDR_ALARM1_SEC	0x0b
#define ADDR_ALARM1_MIN	0x0c
#define ADDR_ALARM1_HOUR	0x0d
#define ADDR_ALARM1_WDAY	0x0e
#define ADDR_CONTROL	0x0f
#define ADDR_STATUS	0x10
#define ADDR_TRICKLE	0x11
#define ADDR_USER_AREA	0x20

#undef DEBUG_RTC
//#define DEBUG_RTC 1

#ifdef DEBUG_RTC
#define DEBUGR(fmt,args...) printf(fmt ,##args)
#else
#define DEBUGR(fmt,args...)
#endif

static int ds1305_write_date(struct rtc_time *date);
static int ds1305_read(unsigned addr, unsigned char *data, int count);
static int ds1305_write(unsigned addr, unsigned char *data, int count);
static int ds1305_set_time(unsigned long);
unsigned char ds1305_read_single(unsigned long offset);
static void ds1305_write_single(unsigned char data, unsigned long offset);

static int spi_param = 0;

static unsigned bcdtodec(unsigned char bcd)
{
	return (bcd >> 4) * 10 + (bcd & 0xf);
}

static unsigned dectobcd(unsigned char dec)
{
	return ((dec / 10) << 4) + (dec % 10);
}

static int ds1305_read(unsigned addr, unsigned char *data, int count)
{
	unsigned char buf[256];

	if (addr + count >= 0x80)
		count = 0x80 - addr;
	if (count <= 0)
		return 0;

	spi_open(spi_param);

	spi_enable();

	udelay(4);

	memset(buf, 0, sizeof(buf));
	buf[0] = addr;
#if 1
	spi_exchange(buf, count + 1);
	memcpy(data, &buf[1], count);
#else
	spi_write(buf, 1);
	spi_read(data, count);
#endif

	spi_disable();

	spi_close();

	return count;
}

static int ds1305_write(unsigned addr, unsigned char *data, int count)
{
	unsigned char buf[256];

	if (addr + count >= 0x80)
		count = 0x80 - addr;
	if (count <= 0)
		return 0;

	spi_open(spi_param);

	spi_enable();

	udelay(4);

	buf[0] = addr | 0x80;
#if 1
	memcpy(&buf[1], data, count);
	spi_write(buf, count + 1);
#else
	spi_write(buf, 1);
	spi_write(data, count);
#endif

	spi_disable();

	spi_close();

	return count;
}

static int ds1305_write_date(struct rtc_time *date)
{
	u_int8_t buf[ADDR_YEAR + 1];
	
	buf[ADDR_SEC] = dectobcd(date->tm_sec);
	buf[ADDR_MIN] = dectobcd(date->tm_min);
	buf[ADDR_HOUR] = dectobcd(date->tm_hour);
	buf[ADDR_WDAY] = dectobcd(date->tm_wday);
	buf[ADDR_DAY] = dectobcd(date->tm_mday);
#if 0
	buf[ADDR_MONTH] = dectobcd(date->tm_mon);
	buf[ADDR_YEAR] = dectobcd(date->tm_year - 1900);
#else
	buf[ADDR_MONTH] = dectobcd(date->tm_mon);
        buf[ADDR_YEAR] = dectobcd(date->tm_year % 100);
#endif

        DEBUGR ("Write DATE: %4d-%02d-%02d (wday=%d)  TIME: %2d:%02d:%02d\n",
                date->tm_year, date->tm_mon, date->tm_mday, date->tm_wday,
                date->tm_hour, date->tm_min, date->tm_sec);
	ds1305_write(0, buf, sizeof(buf));

	return 0;
}

void rtc_get(struct rtc_time *date)
{
	u_int8_t buf[ADDR_YEAR + 1];

	ds1305_read(0, buf, sizeof(buf));
	
	memset(date, 0, sizeof(struct rtc_time));
	date->tm_sec = bcdtodec(buf[ADDR_SEC]);
	date->tm_min = bcdtodec(buf[ADDR_MIN]);
#if 0
	date->tm_hour = bcdtodec(buf[ADDR_HOUR]);
#else
	date->tm_hour = bcdtodec(buf[ADDR_HOUR] & 0x3f);
#endif
	date->tm_mday = bcdtodec(buf[ADDR_DAY]);
	date->tm_mon = bcdtodec(buf[ADDR_MONTH]);
	date->tm_wday = bcdtodec(buf[ADDR_WDAY]);
#if 0
	date->tm_year = bcdtodec(buf[ADDR_YEAR]) + 1900;
#else
	date->tm_year = bcdtodec(buf[ADDR_YEAR]) + 2000;
#endif
        DEBUGR ("Read DATE: %4d-%02d-%02d (wday=%d)  TIME: %2d:%02d:%02d\n",
                date->tm_year, date->tm_mon, date->tm_mday, date->tm_wday,
                date->tm_hour, date->tm_min, date->tm_sec);

	return 0;
}

static int ds1305_set_time(unsigned long t)
{
	struct rtc_time tm;

	to_tm(t, &tm);
	
	ds1305_write_date(&tm);
	
	return 0;
}

int ds1305_init(int param)
{
	unsigned char tmp;
	unsigned long tt;

	spi_param = param;

	ds1305_read(ADDR_CONTROL, &tmp, 1);
	tmp &= ~0x40;	/* clear write protect */
	ds1305_write(ADDR_CONTROL, &tmp, 1);
	tmp = 0x00;	/* enable osillator disable INT, disalbe alarms */
	ds1305_write(ADDR_CONTROL, &tmp, 1);
	tmp = 0xa5;	/* 1 diode, 2kohm */ 
	ds1305_write(ADDR_TRICKLE, &tmp, 1);

	ds1305_read(ADDR_HOUR, &tmp, 1);
	if ((tmp & 0x40)) {
		/* 24 hour mode */
		tmp &= ~0x40;
		ds1305_write(ADDR_HOUR, &tmp, 1);
	}

	return 0;
}

void rtc_set (struct rtc_time *tmp)
{
	unsigned long tmp_time;
        DEBUGR ("Set DATE: %4d-%02d-%02d (wday=%d)  TIME: %2d:%02d:%02d\n",
                tmp->tm_year, tmp->tm_mon, tmp->tm_mday, tmp->tm_wday,
                tmp->tm_hour, tmp->tm_min, tmp->tm_sec);

        if (tmp->tm_year < 1970 || tmp->tm_year > 2069)
                printf("WARNING: year should be between 1970 and 2069!\n");

	tmp_time = mktime(tmp->tm_year, tmp->tm_mon,tmp->tm_mday,
			tmp->tm_hour,tmp->tm_min, tmp->tm_sec);
	ds1305_set_time(tmp_time);
}

void rtc_reset(void)
{
	struct rtc_time tmp;

	tmp.tm_year = 1970;
        tmp.tm_mon = 1;
        tmp.tm_mday= 1;
        tmp.tm_hour = 0;
        tmp.tm_min = 0;
        tmp.tm_sec = 0;

	rtc_set(&tmp);

	printf ( "RTC:   %4d-%02d-%02d %2d:%02d:%02d UTC\n",
                tmp.tm_year, tmp.tm_mon, tmp.tm_mday,
                tmp.tm_hour, tmp.tm_min, tmp.tm_sec);

	return;
}
