/*
 * Adm5120 Ethernet support
 *
 * (C) Copyright 2006 Masami Komiya <mkomiya@sonare.it>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 */
#include <common.h>
#include <malloc.h>
#include <net.h>
#include <asm/io.h>

#if (CONFIG_COMMANDS & CFG_CMD_NET) && defined(CONFIG_NET_MULTI) && \
       defined(CONFIG_ADM5120_ETH)

/* ADM5120 eth sw register base */
#define ADM5120_REG_BASE       0xB2000000

/* write/read eth sw register */
#define ADM_W32(reg, val32)    writel ((val32), dev->iobase + (reg))
#define ADM_R32(reg)           ((unsigned long) readl (dev->iobase + (reg)))

/* Number of Tx/Rx descriptor */
#define NUM_TX_DESC_H          2
#define NUM_TX_DESC_L          16                      
#define NUM_RX_DESC_H          2
#define NUM_RX_DESC_L          32

/* Receive buffer size */
#define RX_BUF_SIZE            1800

#define ETH_FRAME_LEN          MAX_ETH_FRAME_SIZE
#define ETH_ALEN               MAC_ADDR_LEN
#define ETH_ZLEN               60
#define ETH_CRC_LEN            4

#define DEFALT_PORT            0x10

#define TX_TIMEOUT             (6*1000*1000)

enum adm5120_eth_registers {
       Code = 0x0000,
       SftRes = 0x0004,
       Boot_D = 0x0008,
       SW_Res = 0x000C,
       Gl_St = 0x0010,
       PHY_St = 0x0014,
       Port_St = 0x0018,
       Mem_Cont = 0x001C,
       SW_conf = 0x0020,
       CPUp_conf = 0x0024,
       Port_conf0 = 0x0028,
       Port_conf1 = 0x002C,
       Port_conf2 = 0x0030,
       VLAN_G1 = 0x0040,
       VLAN_G2 = 0x0044,
       Send_trig = 0x0048,
       Srch_cmd = 0x004C,
       ADDR_st0 = 0x0050,
       ADDR_st1 = 0x0054,
       MAC_wt0 = 0x0058,
       MAC_wt1 = 0x005C,
       BW_cntl0 = 0x0060,
       BW_cntl1 = 0x0064,
       PHY_cntl0 = 0x0068,
       PHY_cntl1 = 0x006C,
       FC_th = 0x0070,
       Adj_port_th = 0x0074,
       Port_th = 0x0078,
       PHY_cntl2 = 0x007C,
       PHY_cntl3 = 0x0080,
       Pri_cntl = 0x0084,
       VLAN_pri = 0x0088,
       TOS_en = 0x008C,
       TOS_map0 = 0x0090,
       TOS_map1 = 0x0094,
       Custom_pri1 = 0x0098,
       Custom_pri2 = 0x009C,
       PHY_cntl4 = 0x00A0,
       Empty_cnt = 0x00A4,
       Port_cnt_sel = 0x00A8,
       Port_cnt = 0x00AC,
       Int_st = 0x00B0,
       Int_mask = 0x00B4,
       GPIO_conf0 = 0x00B8,
       GPIO_conf2 = 0x00BC,
       Wdog_0 = 0x00C0,
       Wdog_1 = 0x00C4,
       Swap_in = 0x00C8,
       Swap_out = 0x00CC,
       send_Hbaddr = 0x00D0,
       send_Lbaddr = 0x00D4,
       rec_Hbaddr = 0x00D8,
       rec_Lbaddr = 0x00DC,
       send_Hwaddr = 0x00E0,
       send_Lwaddr = 0x00E4,
       rec_Hwaddr = 0x00E8,
       rec_Lwaddr = 0x00EC,
       Timer_int = 0x00F0,
       Timer = 0x00F4,
       port0_LED = 0x0100,
       port1_LED = 0x0104,
       port2_LED = 0x0108,
       port3_LED = 0x010c,
       port4_LED = 0x0110,
};

enum adm5120_eth_register_content {

       Code_PK = 0x20000000,     /* Package type */

       OWN_BIT = 0x80000000,

       END_OF_RING = 0x10000000,       
       
       TX_BUF_ADRS_MASK = 0x01ffffff,
       APPEND_CHKSUM = 0x80000000,
       TX_LEN_SHIFT = 16,
       TX_DEST_PORT_SHIFT = 8,

       RX_BUF_ADRS_MASK = 0x01ffffff,
       RX_PKT_LEN_MASK = 0x07ff0000,
       RX_PKT_LEN_SHIFT = 16,
       CHKSUMERR = 0x00000008,

       /* CPUp_conf */
       DBCP = 0x3f000000,
       DMCP = 0x003f0000,
       DUNP = 0x00007e00,
       CRCP = 0x00000002,
       DCPUP = 0x00000001,

       /* Port_conf0 */
       FC   = 0x3f000000,
       EBP  = 0x003f0000,
       EMCP = 0x00003f00,
       DP   = 0x0000003f,

       /* PHY_cntl2 */
       PHYR = 0x01f00000,
       ANE = 0x0000001f,

       /* MAC_wr0 */
       MWD = 0x00000002,
};

typedef struct {
       u32 buf1cntl;
       u32 buf2cntl;
       u32 buf1len;
       u32 pktcntl;
} TX_DESC;

typedef struct {
       u32 buf1cntl;
       u32 buf2cntl;
       u32 buf1len;
       u32 status;
} RX_DESC;

#define BOOT_LINE_SIZE 255
#define MAC_MAGIC      0x636d676d
#define BOARD_CFG_ADDR 0xbfc08000

typedef struct {
       unsigned long blmagic;
       unsigned char bootline[BOOT_LINE_SIZE+1];
       unsigned long macmagic;
       unsigned char mac[4][8];
} BOARD_CFG;

static TX_DESC tx_desc_h[NUM_TX_DESC_H];
static TX_DESC tx_desc_l[NUM_TX_DESC_L];
static RX_DESC rx_desc_h[NUM_RX_DESC_H];
static RX_DESC rx_desc_l[NUM_RX_DESC_L];

static uchar rxb[NUM_RX_DESC_L][RX_BUF_SIZE];

static int txh_idx, txl_idx;
static int rxh_idx, rxl_idx;

static void init_TxDesc(void)
{
       tx_desc_h[NUM_TX_DESC_H-1].buf1cntl |= END_OF_RING;
       tx_desc_l[NUM_TX_DESC_L-1].buf1cntl |= END_OF_RING;
       txh_idx = 0;
       txl_idx = 0;
}

static void init_RxDesc(void)
{
       int i;

       for (i=0; i<NUM_RX_DESC_H; ++i) {
               rx_desc_h[i].buf1cntl = ((u32)&(rxb[i]) & RX_BUF_ADRS_MASK) | (u32)OWN_BIT;
               rx_desc_h[i].buf2cntl = 0;
               rx_desc_h[i].buf1len  = RX_BUF_SIZE;
               rx_desc_h[i].status   = 0;
       }

       for (i=0; i<NUM_RX_DESC_L; ++i) {
               rx_desc_l[i].buf1cntl = ((u32)&(rxb[i]) & RX_BUF_ADRS_MASK) | (u32)OWN_BIT;
               rx_desc_l[i].buf2cntl = 0;
               rx_desc_l[i].buf1len  = RX_BUF_SIZE;
               rx_desc_l[i].status   = 0;
       }

       rx_desc_h[NUM_RX_DESC_H-1].buf1cntl |= END_OF_RING;
       rx_desc_l[NUM_RX_DESC_L-1].buf1cntl |= END_OF_RING;
       rxh_idx = 0;
       rxl_idx = 0;
}

static void read_MacAdrs(struct eth_device* dev)
{
       BOARD_CFG *bdcfg;

       bdcfg = BOARD_CFG_ADDR;

       if (bdcfg->macmagic == MAC_MAGIC)
               memcpy(dev->enetaddr, bdcfg->mac[0], sizeof(dev->enetaddr));
       else
               memcpy(dev->enetaddr, CONFIG_ADM5120_ETHADDR, sizeof(dev->enetaddr));
}

static void set_MacAdrs(struct eth_device* dev, int vlan)
{
       u32 reg0, reg1;

#define ea eth_get_dev()->enetaddr

       reg0 = ((u32) ea[1] << 24) | ((u32) ea[0] << 16) | (vlan << 5) | (1<<13) | 0x40 | 0x01;
      reg1 = ((u32) ea[5] << 24) | ((u32) ea[4] << 16) |
              ((u32) ea[3] << 8) | (u32) ea[2];

       ADM_W32(MAC_wt1, reg1);
       ADM_W32(MAC_wt0, reg0);

       while (!(ADM_R32(MAC_wt0) & MWD));
}

static int adm5120_init(struct eth_device* dev, bd_t * bd)
{
       u32 val;

       val = ADM_R32(Port_conf0) & (~DP);
       ADM_W32(Port_conf0, val);
//     val = ADM_R32(CPUp_conf) & (~DCPUP);
//     ADM_W32(CPUp_conf, val);

       set_MacAdrs(dev, 0);

       return(1);
}

static void adm5120_halt(struct eth_device* dev)
{
       u32 val;

       val = ADM_R32(Port_conf0) | DP;
       ADM_W32(Port_conf0, val);
//     val = ADM_R32(CPUp_conf) | DCPUP;
//     ADM_W32(CPUp_conf, val);
}

static int adm5120_send(struct eth_device* dev, volatile void *packet, int length)
{
       TX_DESC *tx_desc;
       int len;
       u32 to;

       tx_desc = &(tx_desc_l[txl_idx]);
       len = length < ETH_ZLEN ? ETH_ZLEN : length;

       if (tx_desc->buf1cntl & OWN_BIT) {
               return 0;
       }

       tx_desc->buf1cntl = (tx_desc->buf1cntl & END_OF_RING) |
                           (((unsigned long)packet & TX_BUF_ADRS_MASK) | OWN_BIT);
      tx_desc->buf2cntl = 0;
      tx_desc->buf1len = len;
      tx_desc->pktcntl = (len << TX_LEN_SHIFT) | DEFALT_PORT << TX_DEST_PORT_SHIFT;
     if (++txl_idx >= NUM_TX_DESC_L) txl_idx = 0;

       ADM_W32(Send_trig, 0x01);

       to = get_timer(0) + TX_TIMEOUT;

       while ((tx_desc_l->buf1cntl & OWN_BIT) && (get_timer(0) < to));

       if (get_timer(0) >= to) {
               return 0;
       }
       else {
               return length;
       }
}

static int adm5120_recv(struct eth_device* dev)
{
       RX_DESC *rx_desc;
       int len;

       rx_desc = &rx_desc_l[rxl_idx];

       while (!(rx_desc->buf1cntl & OWN_BIT)) {
               len = ((rx_desc->status & RX_PKT_LEN_MASK) >> RX_PKT_LEN_SHIFT);
               if (len > 0 && !(rx_desc->status & CHKSUMERR)) {
                       NetReceive((uchar *)&(rxb[rxl_idx]), len);
               }

               rx_desc->buf1cntl = (rx_desc->buf1cntl & END_OF_RING ) | ((u32)&(rxb[rxl_idx]) & RX_BUF_ADRS_MASK) | OWN_BIT;
               rx_desc->buf2cntl = 0;
               rx_desc->buf1len  = RX_BUF_SIZE;
               rx_desc->status   = 0;

               if (++rxl_idx == NUM_RX_DESC_L) rxl_idx = 0;

               rx_desc = &rx_desc_l[rxl_idx];
       }

       return 0;
}

static void adm5120_setup(struct eth_device* dev, bd_t *bis)
{
       u32 val;

       init_TxDesc();
       init_RxDesc();

       val = ((ADM_R32(CPUp_conf) & (~DBCP)) & (~DMCP)) | CRCP;
       ADM_W32(CPUp_conf, val);

       val = ADM_R32(PHY_cntl2) | ANE | PHYR;
       ADM_W32(PHY_cntl2, val);

       val = ADM_R32(PHY_cntl3) | (0x0f<<17);
       ADM_W32(PHY_cntl3, val);

       ADM_W32(Int_mask, 0x0fffffff);
       val = ADM_R32(Int_st);
       ADM_W32(Int_st, val);

       ADM_W32(send_Hbaddr, KSEG1ADDR((u32)tx_desc_h));
       ADM_W32(send_Lbaddr, KSEG1ADDR((u32)tx_desc_l));
       ADM_W32(rec_Hbaddr,  KSEG1ADDR((u32)rx_desc_h));
       ADM_W32(rec_Lbaddr,  KSEG1ADDR((u32)rx_desc_l));

       ADM_W32(VLAN_G1, 0x7f);
       ADM_W32(VLAN_G2, 0x00);

       val = ADM_R32(CPUp_conf) & (~DCPUP);
       ADM_W32(CPUp_conf, val);

       read_MacAdrs(dev);

       memcpy(bis->bi_enetaddr, dev->enetaddr, sizeof(dev->enetaddr));
}

int adm5120_eth_initialize(bd_t *bis)
{
      struct eth_device* dev;

      dev = (struct eth_device*) malloc(sizeof *dev);
      memset(dev, 0, sizeof *dev);

      sprintf(dev->name, "ADM5120 ETH");

      dev->priv   = 0;
      dev->iobase = ADM5120_REG_BASE;

      dev->init   = adm5120_init;
      dev->halt   = adm5120_halt;
      dev->send   = adm5120_send;
      dev->recv   = adm5120_recv;

      eth_register(dev);

      adm5120_setup(dev, bis);

      return 1;
}
#endif /* CONFIG_ADM5120_ETH */
