/*
 * adm51xx_eth.c -- Ethernet driver for ADM5120, ADM5106 (untested)
 *                  and possibly more SoCs
 *
 * Copyright (c) 2006 SATO Masuhiro <thomas@fenix.ne.jp>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
/*
 * Usage for this driver is difficult to explain, since ADM5120 Ether
 * net controler has programable HUB.
 *
 * - ADM5120 has 6 HUBs, which makes VLAN.
 * - The software send/receive packets to/from CPU port on the VLANs
 *   instead of physical Ether NET ports.
 *
 * Setting of the HUB can be overrided by environment value, "ethports", like
 *
 * ethports=0x4f,0x50
 *
 * This parameter makes.
 * VLAN0: port0, port1, port2, port3, cpu
 * VLAN1: port4, cpu
 * 
 */

#include <config.h>

#if (CONFIG_COMMANDS & CFG_CMD_NET) && defined(CONFIG_NET_MULTI) && \
	defined(CONFIG_ADM51xx_SWITCH)

#include <common.h>
#include <malloc.h>
#include <net.h>
#include <asm/io.h>
#include <adm51xx_switch.h>

#define RX_HP_BUF_COUNT	4
#define TX_HP_BUF_COUNT	4
#define RX_BUF_COUNT	4
#define TX_BUF_COUNT	4
#define BUFSIZE		1536

#define DESC_OWN_SC	0x80000000
#define DESC_RING_END	0x10000000

//#define DEBUG 1
#undef DEBUG

struct buf_desc {
	u32 buffer1;
	u32 buffer2;
	u32 buffer1_length;
	u32 flags;
} __attribute__((packed));

static int adm5120_eth_init(struct eth_device* dev, bd_t * bd);
static void adm5120_eth_halt(struct eth_device *dev);
static int adm5120_eth_recv(struct eth_device *dev);
static int adm5120_eth_send(struct eth_device *dev, volatile void *packet, int length);
static int adm5120_eth_update(void);

static void init_buffer(void);
static int init_buf_ring(struct buf_desc *buf_desc, int count, u8 *buf, u32 own);
static void adm5120_switch_init(void);
static int tx_hp_update(void);
static int tx_update(void);
static int rx_hp_update(void);
static int rx_update(void);

/* all ports are connected to VLAN0 */
static u8 port_map[SWITCH__VLAN_PORTS] = {
	SWITCH__PORT_CPU | SWITCH__PORT_4 | SWITCH__PORT_3 | SWITCH__PORT_2 | SWITCH__PORT_1 | SWITCH__PORT_0
};
static u32 port_enable;

static int tx_put, tx_done, tx_hp_put, tx_hp_done;
static int rx_put, rx_done, rx_suspend, rx_hp_put, rx_hp_done, rx_hp_suspend;
/* buffers for high priority packets */
static u8 tx_hp_buf[RX_HP_BUF_COUNT * BUFSIZE] __attribute__((aligned(32)));
static u8 rx_hp_buf[TX_HP_BUF_COUNT * BUFSIZE] __attribute__((aligned(32)));
/* buffers for normal priority packets */
static u8 tx_buf[RX_BUF_COUNT * BUFSIZE] __attribute__((aligned(32)));
static u8 rx_buf[TX_BUF_COUNT * BUFSIZE] __attribute__((aligned(32)));

struct buf_desc	rx_hp_desc[RX_HP_BUF_COUNT] __attribute__((aligned(32)));
struct buf_desc	tx_hp_desc[TX_HP_BUF_COUNT] __attribute__((aligned(32)));
;
struct buf_desc	rx_desc[RX_BUF_COUNT] __attribute__((aligned(32)));
;
struct buf_desc	tx_desc[TX_BUF_COUNT] __attribute__((aligned(32)));


int adm5120_eth_initialize(bd_t *bis)
{
	char *p;
	int i;

	if ((p = getenv("ethports")) != NULL) {
		long ports;

		i = 0;
		memset(port_map, 0, sizeof(port_map));
		while (*p) {
			ports = simple_strtol(p, &p, 0x10);
			if (ports)
				port_map[i++] = (u8)ports;
			if (*p == 0)
				break;
			if (*p != ',' && *p != ':')
				break;
			p++;
		}
	}

	adm5120_switch_init();

#ifdef DEBUG
	printf("adm5120_eth port groups ");
#endif
	for (i = 0; i < SWITCH__VLAN_PORTS && port_map[i] != 0; i++) {
		struct eth_device *dev;
#ifdef DEBUG
		int	j;
		u8	bit = 0x01;

		printf("%d:", i);

		for (j = 0; j < SWITCH__PHYS_PORTS + 2; j++) {
			if (bit & port_map[i]) {
				if (j > 0)
					printf(".", j);
				printf("%d", j);
			}
			bit <<= 1;
		}
		printf(" ");
#endif

		dev = (struct eth_device *)malloc(sizeof(struct eth_device));
		if (dev == NULL) {
			printf("\nadm5120_eth_initialize(): malloc() failed\n");
			break;
		}
		memset(dev, 0, sizeof(struct eth_device));

		sprintf(dev->name, "adm%d", i);
		dev->iobase = 0;
		dev->priv   = (void *)i;
		dev->init   = adm5120_eth_init;
		dev->halt   = adm5120_eth_halt;
		dev->send   = adm5120_eth_send;
		dev->recv   = adm5120_eth_recv;

		eth_register(dev);
	}

	return 1;
}

static void adm5120_switch_init()
{
	int	i;

	writel( SWITCH_CPUP_CONF__CRCP,
		SWITCH_CPUP_CONF + CONFIG_ADM51xx_SWITCH_BASE );
	writel( SWITCH_PORT_CONF0__DP,
		SWITCH_PORT_CONF0 + CONFIG_ADM51xx_SWITCH_BASE );

	barrier();
	udelay(10000);

	/*
	 * VLAN Gi 0xB20000040
	 * VLAN0 : VLAN Group 0 6:0
	 * VLAN1 : VLAN Group 1 14:8
	 * VLAN2 : VLAN Group 2 22:16
	 * VLAN3 : VLAN Group 3 30:24
	 */
	writel( SWITCH_VLAN_GI__VLAN0_SET(port_map[0]) |
		SWITCH_VLAN_GI__VLAN1_SET(port_map[1]) |
		SWITCH_VLAN_GI__VLAN2_SET(port_map[2]) |
		SWITCH_VLAN_GI__VLAN3_SET(port_map[3]),
		SWITCH_VLAN_GI + CONFIG_ADM51xx_SWITCH_BASE );

	/*
	 * VLAN Gi 0xB20000044
	 * VLAN4 : VLAN Group 0 6:0
	 * VLAN5 : VLAN Group 0 14:8
	 */
	writel( SWITCH_VLAN_GII__VLAN4_SET(port_map[4]) |
		SWITCH_VLAN_GII__VLAN5_SET(port_map[5]),
		SWITCH_VLAN_GII + CONFIG_ADM51xx_SWITCH_BASE );

	barrier();

	port_enable = 0;
	for (i = 0; i < SWITCH__VLAN_PORTS; i++)
		port_enable |= port_map[i];

#ifdef DEBUG
	printf("port_enable = %08x\n", port_enable);
#endif

	/*
	 * CPUp Conf 0xB2000024
	 * DCPUP: CPU Port Disable     Bit0
	 * CRCP : CRC Padding from CPU Bit1
	 * BTM  : Bridge testing Mode  Bit2
	 * DUNP : Disable Unknown Packets from Ports, Forward to CPU Bit14:9
	 * DMCP : Disable Multicast Packets, from port , Forward to CPU Bit21:16
	 * DBCP : Disable Broadcast Packets, from port, Forward to CPU Bit29:24
	 */
	writel( SWITCH_CPUP_CONF__DCPUP |
		SWITCH_CPUP_CONF__CRCP |
		SWITCH_CPUP_CONF__DUNP_SET(SWITCH__PORTS_NET),
		SWITCH_CPUP_CONF + CONFIG_ADM51xx_SWITCH_BASE );

	/* enable ports, multicast, back pressure */
	/* Port Conf 0xB2000028
	 * DP_SET : Disable Port  Bit5:0
	 * EMCP : Enable All MC Packet Bit13:8
	 * EBP : Enable Back Pressure  Bit21:16
	 */
	writel( SWITCH_PORT_CONF0__DP_SET(~port_enable) |
		SWITCH_PORT_CONF0__EMCP_SET(port_enable) |
		SWITCH_PORT_CONF0__EBP_SET(port_enable),
		SWITCH_PORT_CONF0 + CONFIG_ADM51xx_SWITCH_BASE );

	/*
	 * Port Conf1 0xB200002C
	 * DISL : Dis Learn Bit5:0
	 * BS : Blocking State Bit11:6
	 * BM : Blocking Mode  Bit17:12
	 * PA : Port Aging     Bit25:20
	 * SASM : SA Secured Mode Bit31:26
	 */
	writel( SWITCH_PORT_CONF1__SASM_SET(0) |
		SWITCH_PORT_CONF1__PA_SET(port_enable) |
		SWITCH_PORT_CONF1__BM_SET(0) |
		SWITCH_PORT_CONF1__BS_SET(0) |
		SWITCH_PORT_CONF1__DISL_SET(0),
		SWITCH_PORT_CONF1 + CONFIG_ADM51xx_SWITCH_BASE );

	/*
	 * Port Conf2 0xB2000030
	 * GAIIAN : MII Port AN Monitor Enable Bit0
	 * FMPS : Force MII Port Speed if AN Monitor Disable  Bit2:1
	 * FMDP : Force MII Port Duplex if AN Monitor Disable Bit3
	 * FMPFC: Force MII port FC Bit5:4
	 * RMIIM: Reversed MII Mode Bit7
	 * TMCC : TXC Check         Bit8
	 * CER  : Configure Early Ready Bit10:9
	 * LEDFT: The Frequency of LED Flash Bit17:16
	 * DUPF : Disable Unicast Pause Frame 5:0 Bit23:18
	 */
	if (port_enable & SWITCH__PORT_GIGA) {
		writel( SWITCH_PORT_CONF2__GMIIAN |	/* AN enable */
			SWITCH_PORT_CONF2__FMPS_100M |	/* 100M */	
			SWITCH_PORT_CONF2__FMDP |	/* full duplex */
			SWITCH_PORT_CONF2__FMPFC_RXTX,	/* flow control */
			SWITCH_PORT_CONF2 + CONFIG_ADM51xx_SWITCH_BASE );
	} else {
		writel( SWITCH_PORT_CONF2__FMPS_100M |	/* 100M */	
			SWITCH_PORT_CONF2__FMDP |	/* full duplex */
			SWITCH_PORT_CONF2__FMPFC_RXTX |	/* flow control */
			SWITCH_PORT_CONF2__TXCC,	/* disable check TXC */
			SWITCH_PORT_CONF2 + CONFIG_ADM51xx_SWITCH_BASE );
	}

	barrier();
	udelay(10000);

	/*
	 * PHY Control 2 0xB200007C
	 * ANE  : Auto Negotiation Enable Bit4:0
	 * SC   : Speed Control 0/10M 1/100M Bit9:5
	 * DC   : Duplex Control 0/Half 1/Full Bit14:10
	 * RFCV : Recommended FC Value Bit19:15
	 * PHYR : PHy Reset 0/Reset 1/Normal Bit24:20
	 * AMDIX:Auto MDIX enable Bit29:25
	 * RMAE : Recommend MCC Average Enable Bit30
	 */

	writel( SWITCH_PHY_CNTL2__ANE_SET(port_enable) |	/* auto negotiation */
		SWITCH_PHY_CNTL2__SC_SET(port_enable) |		/* speed control */
		SWITCH_PHY_CNTL2__DC_SET(port_enable) |		/* duplex control */
		SWITCH_PHY_CNTL2__RFCV_SET(port_enable) |	/* FC rec */
		SWITCH_PHY_CNTL2__AMDIX_SET(port_enable) |	/* MDIX */
		SWITCH_PHY_CNTL2__RES1 |			/* Bit 31 */
		SWITCH_PHY_CNTL2__PHYR_SET(port_enable), 	/* disable reset */
		SWITCH_PHY_CNTL2 + CONFIG_ADM51xx_SWITCH_BASE );

	/*
	 * PHY Control 3 0xB2000080
	 * RBLL  : Recommend base line limit     Bit1:0
	 * PFRV  : Pre filter Recommend value    Bit3:2
	 * RSHC  : Recommend Sample Hole Current Bit5:4
	 * IINSEL: IINSEL Bit6
	 * RAPD  : Recommend Auto Polarity Disable   Bit7
	 * RRJE  : Recommend Receive Jabber enable   Bit8
	 * RTJD  : Recommend Transmit Jabber Disable Bit9
	 * RNT   : Recommend Normal Threshold Bit10
	 * RFGL  : Recommend Force Good Link Bit11
	 * RPLFT : Recommend Polarity Link Fail Timer Select Bit13:12
	 * RPIC  : Recommend Polarity LIU/LID Interval Check Bit14
	 * CBDE  : Recommend Cable Broken Detect Enable Bit15
	 * DFEFI : Disable far_end Fault Indication Bit16
	 * FXE   : Per port FX Enable Bit21:17
	 */
#if 1
	writel( SWITCH_PHY_CNTL3__DFEFI |
		SWITCH_PHY_CNTL3__RPIC |
		SWITCH_PHY_CNTL3__RRJE |
		SWITCH_PHY_CNTL3__RBLL_VAL(0) |
		SWITCH_PHY_CNTL3__PFRV_VAL(2) |
		SWITCH_PHY_CNTL3__RBLL_VAL(3),
		SWITCH_PHY_CNTL3 + CONFIG_ADM51xx_SWITCH_BASE );
#else
	writel( SWITCH_PHY_CNTL3__RPIC |
		SWITCH_PHY_CNTL3__RRJE |
		SWITCH_PHY_CNTL3__RBLL_VAL(0) |
		SWITCH_PHY_CNTL3__PFRV_VAL(2) |
		SWITCH_PHY_CNTL3__RBLL_VAL(3),
		SWITCH_PHY_CNTL3 + CONFIG_ADM51xx_SWITCH_BASE );
#endif
	/* Mask all interrupts */
	/*
	 * Interrupt Mask 0xB20000B4
	 */
	writel( SWITCH_INT_MASK__ALL_INTS,
		SWITCH_INT_MASK + CONFIG_ADM51xx_SWITCH_BASE );

	/* Clear all pending interrupts 0xB20000B0 */
	writel( readl(SWITCH_INT_ST + CONFIG_ADM51xx_SWITCH_BASE),
		SWITCH_INT_ST + CONFIG_ADM51xx_SWITCH_BASE );

	barrier();
}

static int init_buf_ring(struct buf_desc *buf_desc, int count, u8 *buf, u32 own)
{
	int i;

	//memset(buf, 0, BUFSIZE * count);
	for (i = 0; i < count; i++) {
		buf_desc[i].buffer1 = own |
			((i == count - 1) ? DESC_RING_END : 0) |
			virt_to_phys(buf + (i * BUFSIZE));
		buf_desc[i].buffer2 = 0;
		buf_desc[i].buffer1_length = BUFSIZE;
		buf_desc[i].flags = 0;
	}
	dma_cache_wback_inv((unsigned long)buf,      BUFSIZE * count);
	dma_cache_wback_inv((unsigned long)buf_desc, sizeof(struct buf_desc) * count);
	
	return i;
}

static void init_buffer()
{
	tx_put = 0;
	tx_done = 0;
	tx_hp_put = 0;
	tx_hp_done = 0;
	rx_put = 0;
	rx_done = 0;
	rx_suspend = RX_BUF_COUNT;
	rx_hp_put = 0;
	rx_hp_done = 0;
	rx_hp_suspend = RX_HP_BUF_COUNT;

	init_buf_ring(tx_hp_desc, TX_HP_BUF_COUNT, tx_hp_buf, 0);
	init_buf_ring(rx_hp_desc, RX_HP_BUF_COUNT, rx_hp_buf, DESC_OWN_SC);
	init_buf_ring(tx_desc,    TX_BUF_COUNT,    tx_buf,    0);
	init_buf_ring(rx_desc,    RX_BUF_COUNT,    rx_buf,    DESC_OWN_SC);

	writel(virt_to_phys(tx_hp_desc), CONFIG_ADM51xx_SWITCH_BASE + SWITCH_SEND_HBADDR);
	writel(virt_to_phys(tx_desc),    CONFIG_ADM51xx_SWITCH_BASE + SWITCH_SEND_LBADDR);
	writel(virt_to_phys(rx_hp_desc), CONFIG_ADM51xx_SWITCH_BASE + SWITCH_REC_HBADDR);
	writel(virt_to_phys(rx_desc),    CONFIG_ADM51xx_SWITCH_BASE + SWITCH_REC_LBADDR);

	barrier();
}

static int adm5120_eth_init(struct eth_device *dev, bd_t *bd)
{
#ifdef DEBUG
	printf( "adm5120_eth_init(): priv=%p dev->enetaddr=%02x:%02x:%02x:%02x:%02x:%02x\n",
		dev->priv,
		dev->enetaddr[0],
		dev->enetaddr[1],
		dev->enetaddr[2],
		dev->enetaddr[3],
		dev->enetaddr[4],
		dev->enetaddr[5] );
#endif
	init_buffer();

	/* Enable CPU port */
	writel( readl(SWITCH_CPUP_CONF + CONFIG_ADM51xx_SWITCH_BASE) & ~SWITCH_CPUP_CONF__DCPUP,
		SWITCH_CPUP_CONF + CONFIG_ADM51xx_SWITCH_BASE );

	writel( SWITCH_MAC_WT1__MWA_B5(dev->enetaddr[5]) |
		SWITCH_MAC_WT1__MWA_B4(dev->enetaddr[4]) |
		SWITCH_MAC_WT1__MWA_B3(dev->enetaddr[3]) |
		SWITCH_MAC_WT1__MWA_B2(dev->enetaddr[2]),
		SWITCH_MAC_WT1 + CONFIG_ADM51xx_SWITCH_BASE );

	writel( SWITCH_MAC_WT0__MWA_B1(dev->enetaddr[1]) |
		SWITCH_MAC_WT0__MWA_B0(dev->enetaddr[0]) |
		SWITCH_MAC_WT0__WAF_SET(1) |
		SWITCH_MAC_WT0__WVE |
		SWITCH_MAC_WT0__WVN_SET((unsigned)dev->priv) |
		SWITCH_MAC_WT0__MAWC,
		SWITCH_MAC_WT0 + CONFIG_ADM51xx_SWITCH_BASE );

	barrier();

	/* Wait for the MAC address write to complete */
	while ((readl(SWITCH_MAC_WT0 + CONFIG_ADM51xx_SWITCH_BASE) & SWITCH_MAC_WT0__MWD) == 0)
		;

	/* Mask all interrupts */
	writel( SWITCH_INT_MASK__ALL_INTS,
		SWITCH_INT_MASK + CONFIG_ADM51xx_SWITCH_BASE );

	/* Clear all pending interrupts */
	writel( readl(SWITCH_INT_ST + CONFIG_ADM51xx_SWITCH_BASE),
		SWITCH_INT_ST + CONFIG_ADM51xx_SWITCH_BASE );

	barrier();

	return(1);
}

static void adm5120_eth_halt(struct eth_device *dev)
{
	/* Disable CPU port */
	writel( readl(SWITCH_CPUP_CONF + CONFIG_ADM51xx_SWITCH_BASE) | SWITCH_CPUP_CONF__DCPUP,
		SWITCH_CPUP_CONF + CONFIG_ADM51xx_SWITCH_BASE );

	barrier();
	udelay(10000);
}

static int adm5120_eth_send(struct eth_device *dev, volatile void *packet, int length)
{
	u8 *bp;
	int retry;
	struct buf_desc *dp;

	retry =  100;
	while (tx_put - tx_done == TX_BUF_COUNT && --retry) {
		udelay(1000);
		adm5120_eth_update();
	}
	if (retry == 0)
		return 0;

	bp = tx_buf + BUFSIZE * (tx_put % TX_BUF_COUNT);
	memcpy(bp, packet, length);

	dp = &tx_desc[tx_put % TX_BUF_COUNT];
	dp->buffer1_length = length;
	if (length < 60)
		length = 60;
	dp->flags = (length << 16) | (1 << (unsigned)dev->priv);
	dp->buffer1 |= DESC_OWN_SC;

	dma_cache_wback_inv((unsigned long)bp, length);
	dma_cache_wback_inv((unsigned long)dp, sizeof(struct buf_desc));
	writel( SWITCH_SEND_TRIG__STL,
		SWITCH_SEND_TRIG + CONFIG_ADM51xx_SWITCH_BASE );
	barrier();

#ifdef DEBUG
	printf( "adm5120_eth_tx(), tx_put=%d, desc=%p, buf=%p, len=%d\n",
		tx_put, dp, bp, length );
#endif

	tx_put++;

	return length;
}

static int adm5120_eth_recv(struct eth_device *dev)
{
	volatile struct buf_desc *dp;
	u8 *bp;
	int len;

	len = 0;

	if (rx_put == rx_done && rx_hp_put == rx_hp_done)
		adm5120_eth_update();

	 while (rx_hp_put > rx_hp_done) {
		bp = rx_hp_buf + BUFSIZE * (rx_hp_done % RX_BUF_COUNT);
		dp = &rx_hp_desc[rx_hp_done % RX_BUF_COUNT];
		
		len = dp->flags >> 16; 
		if (len > 0) {
			NetReceive(bp, len);
			dma_cache_inv((unsigned long)bp, len);
		}
		dp->buffer1 |= DESC_OWN_SC;
		dma_cache_wback_inv((unsigned long)dp, sizeof(struct buf_desc));
		rx_hp_done++;	
		rx_hp_suspend++;	
#ifdef DEBUG
		printf( "adm5120_eth_rx() pull %d bytes. rx_hp_done=%d\n",
			len, rx_hp_done );
#endif
	} 

	while (rx_put > rx_done) {
		bp = rx_buf + BUFSIZE * (rx_done % RX_BUF_COUNT);
		dp = &rx_desc[rx_done % RX_BUF_COUNT];
		
		len = dp->flags >> 16; 
		if (len > 0) {
			NetReceive(bp, len);
			dma_cache_inv((unsigned long)bp, len);
		}
		dp->buffer1 |= DESC_OWN_SC;
		dma_cache_wback_inv((unsigned long)dp, sizeof(struct buf_desc));
		rx_done++;	
		rx_suspend++;	
#ifdef DEBUG
		printf( "adm5120_eth_rx() pull %d bytes. rx_done=%d\n",
			len, rx_done );
#endif
	}

	return 0;
}

static int rx_update()
{
	int	n = 0;

	while (rx_put < rx_suspend) {
		volatile struct buf_desc *dp;
		dp = &rx_desc[rx_put % RX_BUF_COUNT];
		dma_cache_inv((unsigned long)dp, sizeof(struct buf_desc));
		if ((dp->buffer1 & DESC_OWN_SC) != 0)
			break;
		rx_put++;
		n++;
#ifdef DEBUG
		printf( "rx_update() rx_put=%0d, rx_suspend=%d\n",
			rx_put, rx_suspend );
#endif
	}

	return n;
}

static int rx_hp_update()
{
	int	n = 0;

	while (rx_hp_put < rx_hp_suspend) {
		volatile struct buf_desc *dp;
		dp = &rx_hp_desc[rx_put % RX_BUF_COUNT];
		dma_cache_inv((unsigned long)dp, sizeof(struct buf_desc));
		if ((dp->buffer1 & DESC_OWN_SC) != 0)
			break;
		rx_hp_put++;
		n++;
	}

	return n;
}

static int tx_update()
{
	int	n = 0;

	while (tx_put > tx_done) {
		volatile struct buf_desc *dp;

		dp = &tx_desc[rx_put % RX_BUF_COUNT];
		dma_cache_inv((unsigned long)dp, sizeof(struct buf_desc));
		if ((dp->buffer1 & DESC_OWN_SC) != 0)
			break;
		tx_done++;
		n++;
#ifdef DEBUG
		printf( "tx_update() tx_put=%0d, tx_done=%d\n",
			tx_put, tx_done );
#endif
	}
	if (tx_put > tx_done) {
		writel( SWITCH_SEND_TRIG__STL,
			SWITCH_SEND_TRIG + CONFIG_ADM51xx_SWITCH_BASE );
		barrier();
	}

	return n;
}

static int tx_hp_update()
{
	int	n = 0;

	while (tx_hp_put > tx_hp_done) {
		volatile struct buf_desc *dp;

		dp = &tx_hp_desc[rx_put % RX_BUF_COUNT];
		dma_cache_inv((unsigned long)dp, sizeof(struct buf_desc));
		if ((dp->buffer1 & DESC_OWN_SC) != 0)
			break;
		tx_hp_done++;
		n++;
#ifdef DEBUG
		printf( "tx_hp_update() tx_hp_put=%0d, tx_hp_done=%d\n",
			tx_hp_put, tx_hp_done );
#endif
	}

	return n;
}

static int adm5120_eth_update()
{
	int	n = 0;

	n += rx_hp_update();
	n += rx_update();
	n += tx_hp_update();
	n += tx_update();
	
	return n;
}

#endif
